<?php
/* --------------------------------------------------------------
  ConfigWriteService.inc.php 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

namespace StyleEdit;

use \StyleEdit\Factories\ConfigFactory;
use \StyleEdit\Repositories\ConfigRepository;

/**
 * Class ConfigWriteService
 *
 * @package StyleEdit
 */
class ConfigWriteService
{
	/** @var \StyleEdit\Repositories\ConfigRepository $configRepository */
	private $configRepository;
	
	/** @var \StyleEdit\Factories\ConfigFactory $configFactory */
	private $configFactory;
	
	
	/**
	 * ConfigWriteService constructor.
	 *
	 * @param \StyleEdit\Repositories\ConfigRepository $configRepository
	 * @param \StyleEdit\Factories\ConfigFactory       $configFactory
	 */
	public function __construct(ConfigRepository $configRepository, ConfigFactory $configFactory)
	{
		$this->configRepository = $configRepository;
		$this->configFactory    = $configFactory;
	}
	
	
	/**
	 * Creates a new style configuration from a boilerplate configuration. A StoredConfig object will be
	 * returned.
	 *
	 * @param string $p_boilerplateName
	 * @param string $p_styleName
	 *
	 * @throws \InvalidArgumentException if $p_boilerplateName is not a string
	 * @throws \InvalidArgumentException if $p_boilerplateName is an empty string
	 * @throws \InvalidArgumentException if $p_styleName is not a string
	 * @throws \InvalidArgumentException if $p_styleName is an empty string
	 *
	 * @return \StyleEdit\Entities\StoredConfig
	 */
	public function createStyleConfig($p_boilerplateName, $p_styleName)
	{
		if(!is_string($p_boilerplateName))
		{
			throw new \InvalidArgumentException('$p_boilerplateName is not a string: ' . gettype($p_boilerplateName));
		}
		
		if($p_boilerplateName === '')
		{
			throw new \InvalidArgumentException('$p_boilerplateName must not be an empty string');
		}
		
		if(!is_string($p_styleName))
		{
			throw new \InvalidArgumentException('$p_styleName is not a string: ' . gettype($p_styleName));
		}
		
		if($p_styleName === '')
		{
			throw new \InvalidArgumentException('$p_styleName must not be an empty string');
		}
		
		$boilerplate = $this->configRepository->getBoilerplateByName($p_boilerplateName);
		$styleConfig = $this->configFactory->createConfig($boilerplate->getJsonDataArray());
		$styleConfig->setName($p_styleName);
		$styleConfig->setBoilerplateModificationDate($boilerplate->getModificationDate());
		
		if(StyleThemeControl::isThemeSystemActive() && strpos($p_styleName, '__temporary_style') === false)
		{
			$this->configRepository->store($styleConfig, true);
		}
		
		return $this->configRepository->store($styleConfig);
	}
	
	
	/**
	 * Creates a copy of an existing style configuration with a new given name. A StoredConfig object will be
	 * returned.
	 *
	 * @param string $p_sourceStyleName
	 * @param string $p_styleName
	 *
	 * @throws \InvalidArgumentException if $p_sourceStyleName is not a string
	 * @throws \InvalidArgumentException if $p_sourceStyleName is an empty string
	 * @throws \InvalidArgumentException if $p_styleName is not a string
	 * @throws \InvalidArgumentException if $p_styleName is an empty string
	 *
	 * @return \StyleEdit\Entities\StoredConfig
	 */
	public function copyStyleConfig($p_sourceStyleName, $p_styleName)
	{
		if(!is_string($p_sourceStyleName))
		{
			throw new \InvalidArgumentException('$p_sourceStyleName is not a string: ' . gettype($p_sourceStyleName));
		}
		
		if($p_sourceStyleName === '')
		{
			throw new \InvalidArgumentException('$p_sourceStyleName must not be an empty string');
		}
		
		if(!is_string($p_styleName))
		{
			throw new \InvalidArgumentException('$p_styleName is not a string: ' . gettype($p_styleName));
		}
		
		if($p_styleName === '')
		{
			throw new \InvalidArgumentException('$p_styleName must not be an empty string');
		}
		
		$sourceConfig = $this->configRepository->getStyleConfigByName($p_sourceStyleName);
		$styleConfig  = $this->configFactory->createConfig($sourceConfig->getJsonDataArray());
		$styleConfig->setName($p_styleName);
		
		if(StyleThemeControl::isThemeSystemActive())
		{
			$this->configRepository->store($styleConfig, true);
		}
		
		return $this->configRepository->store($styleConfig);
	}
	
	
	/**
	 * Updates a style configuration, identified by its name, by replacing its content. The new content is given in the
	 * second parameter as a JSON formatted string. The updated style configuration will be returned as a
	 * StoredConfig object.
	 *
	 * @param string $p_styleName
	 * @param string $p_jsonString
	 *
	 * @throws \InvalidArgumentException if $p_styleName is not a string
	 * @throws \InvalidArgumentException if $p_styleName is an empty string
	 * @throws \InvalidArgumentException if $p_jsonString is not a JSON formatted string
	 *
	 * @return \StyleEdit\Entities\StoredConfig
	 */
	public function updateStyleConfig($p_styleName, $p_jsonString)
	{
		if(!is_string($p_styleName))
		{
			throw new \InvalidArgumentException('$p_styleName is not a string: ' . gettype($p_styleName));
		}
		
		if($p_styleName === '')
		{
			throw new \InvalidArgumentException('$p_styleName must not be an empty string');
		}
		
		try
		{
			$jsonDataArray = JsonTransformer::decode($p_jsonString);
		}
		catch(\RuntimeException $e)
		{
			throw new \InvalidArgumentException('$p_jsonString is not a JSON formatted string. ' . $e->getMessage());
		}
		
		$styleConfig = $this->configRepository->getStyleConfigByName($p_styleName);
		$styleConfig->setJsonDataArray($jsonDataArray);
		
		if(StyleThemeControl::isThemeSystemActive() && strpos($p_styleName, '__temporary_style') === false)
		{
			$this->configRepository->store($styleConfig, true);
		}
		
		return $this->configRepository->store($styleConfig);
	}
	
	
	/**
	 * Deletes a style configuration identified by its name. If the deletion was successful true will be returned,
	 * otherwise false.
	 *
	 * @param string $p_styleName
	 *
	 * @throws \InvalidArgumentException if $p_styleName is not a string
	 * @throws \InvalidArgumentException if $p_styleName is an empty string
	 *
	 * @return bool
	 */
	public function deleteStyleConfigByName($p_styleName)
	{
		if(!is_string($p_styleName))
		{
			throw new \InvalidArgumentException('$p_styleName is not a string: ' . gettype($p_styleName));
		}
		
		if($p_styleName === '')
		{
			throw new \InvalidArgumentException('$p_styleName must not be an empty string');
		}
		
		if(StyleThemeControl::isThemeSystemActive())
		{
			$this->configRepository->deleteStyleConfigByName($p_styleName, true);
		}
		
		return $this->configRepository->deleteStyleConfigByName($p_styleName);
	}
	
	
	/**
	 * Renames a style configuration identified by its current (old) name. The updated style configuration will be
	 * returned as a StoredConfig object.
	 *
	 * @param $p_oldStyleName
	 * @param $p_newStyleName
	 *
	 * @throws \InvalidArgumentException if $p_oldStyleName is not a string
	 * @throws \InvalidArgumentException if $p_oldStyleName is an empty string
	 * @throws \InvalidArgumentException if $p_newStyleName is not a string
	 * @throws \InvalidArgumentException if $p_newStyleName is an empty string
	 *
	 * @return \StyleEdit\Entities\StoredConfig
	 */
	public function renameStyleConfig($p_oldStyleName, $p_newStyleName)
	{
		if(!is_string($p_oldStyleName))
		{
			throw new \InvalidArgumentException('$p_oldStyleName is not a string: ' . gettype($p_oldStyleName));
		}
		
		if($p_oldStyleName === '')
		{
			throw new \InvalidArgumentException('$p_oldStyleName must not be an empty string');
		}
		
		if(!is_string($p_newStyleName))
		{
			throw new \InvalidArgumentException('$p_newStyleName is not a string: ' . gettype($p_newStyleName));
		}
		
		if($p_newStyleName === '')
		{
			throw new \InvalidArgumentException('$p_newStyleName must not be an empty string');
		}
		
		$storedConfig = $this->configRepository->getStyleConfigByName($p_oldStyleName);
		$storedConfig->setName($p_newStyleName);
		
		if(StyleThemeControl::isThemeSystemActive())
		{
			$this->configRepository->store($storedConfig, true);
		}
		
		return $this->configRepository->store($storedConfig);
	}
	
	
	/**
	 * Creates a new style configuration from a JSON string input and returns a StoredConfig object.
	 *
	 * @param string $p_styleName
	 * @param string $p_jsonString
	 *
	 * @throws \InvalidArgumentException if $p_styleName is not a string
	 * @throws \InvalidArgumentException if $p_styleName is an empty string
	 * @throws \InvalidArgumentException if $p_jsonString is not a JSON formatted string
	 *
	 * @return \StyleEdit\Entities\StoredConfig
	 */
	public function uploadStyleConfig($p_styleName, $p_jsonString)
	{
		try
		{
			$jsonDataArray = JsonTransformer::decode($p_jsonString);
		}
		catch(\RuntimeException $e)
		{
			throw new \InvalidArgumentException('$p_jsonString is not a JSON formatted string. ' . $e->getMessage());
		}
		
		if(!is_string($p_styleName))
		{
			throw new \InvalidArgumentException('$p_styleName is not a string: ' . gettype($p_styleName));
		}
		
		if($p_styleName === '')
		{
			throw new \InvalidArgumentException('$p_styleName must not be an empty string');
		}
		
		$styleConfig = $this->configFactory->createConfig($jsonDataArray);
		$styleConfig->setName($p_styleName);
		
		if(StyleThemeControl::isThemeSystemActive())
		{
			$this->configRepository->store($styleConfig, true);
		}
		
		return $this->configRepository->store($styleConfig);
	}
	
	
	/**
	 * Activates a StyleConfig identified by its name. All other StyleConfigs will be deactivated. The activated
	 * StyleConfig will be returned.
	 *
	 * @param $p_styleName
	 *
	 * @return \StyleEdit\Entities\StoredConfig
	 *
	 * @throws \InvalidArgumentException if $p_styleName is not a string
	 * @throws \InvalidArgumentException if $p_styleName is an empty string
	 */
	public function activateStyle($p_styleName)
	{
		if(!is_string($p_styleName))
		{
			throw new \InvalidArgumentException('$p_styleName is not a string: ' . gettype($p_styleName));
		}
		
		if($p_styleName === '')
		{
			throw new \InvalidArgumentException('$p_styleName must not be an empty string');
		}
		
		$styleConfig = $this->configRepository->getStyleConfigByName($p_styleName);
		
		while($activeStyleConfig = $this->configRepository->findActiveStyleConfig())
		{
			$activeStyleConfig->deactivate();
			
			if(StyleThemeControl::isThemeSystemActive())
			{
				$this->configRepository->store($activeStyleConfig, true);
			}
			
			$this->configRepository->store($activeStyleConfig);
		}
		
		$styleConfig->activate();
		
		if(StyleThemeControl::isThemeSystemActive())
		{
			$this->configRepository->store($styleConfig, true);
		}
		
		return $this->configRepository->store($styleConfig);
	}
}
